<?php


namespace App\Services;


use App\Mail\Verification;
use App\Models\PasswordReset;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Throwable;

class AccountVerificationService
{

    private $mailService;
    private $smsService;
    private $hashService;

    public function __construct(MailService $mailService, SmsService $smsService, HashService $hashService)
    {
        $this->mailService = $mailService;
        $this->smsService = $smsService;
        $this->hashService = $hashService;
    }

    public function create(Model $user)
    {
        if ($user == null) {
            throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_credential')], 400));
        }
        $token = Str::random(50) . uniqid("", true) . Str::random(50) . time();
        $token = preg_replace('/[^a-zA-Z0-9_ -]/s', '', $token);
        $latestAttempt = PasswordReset::query()->where('email', '=', $user->email)->latest()->first();
        if ($latestAttempt != null && Carbon::now()->diffInRealMinutes(Carbon::parse($latestAttempt->created_at), true) < 2) {
            throw new HttpResponseException(response()->json(['error' => __('common.error_wait_2_min')], 400));
        }
        PasswordReset::query()->where('email', '=', $user->email)->forceDelete();
        $verificationCode = $this->hashService->numericHash(time(), 6);
        PasswordReset::query()->create([
            'email' => $user->email,
            'token' => $token,
            'verification_code' => $verificationCode,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ]);
        $url = config('app.url') . '/api/verifyAccount?';
        $url = $url . 'token=' . $token;
        $url = $url . '&';
        $url = $url . 'email=' . $user->email;
        $this->mailService->sendToUser($user, new Verification([
            'first_name' => $user->first_name,
            'last_name' => $user->last_name,
            'code' => $verificationCode
        ], $url));
        $this->smsService->send($user->phone, $verificationCode, $user->locale);
    }

    public function verifyByCode($code, $phone, $email): JsonResponse
    {
        try {
            DB::beginTransaction();
            if (is_null($code) || $code == '') {
                throw new HttpResponseException(Response::failure(__('common.error_500_account_verification')));
            }
            $user = User::phoneOrEmail($phone, $email)->first();
            if (is_null($user)) {
                throw new HttpResponseException(Response::failure(__('common.error_500_account_verification')));
            }
            $verifyInfo = PasswordReset::query()
                ->where('verification_code', '=', $code)
                ->where('email', '=', $user->email)
                ->first();
            if (is_null($verifyInfo)) {
                throw new HttpResponseException(Response::failure(__('common.error_500_account_verification')));
            }
            $user->update(['verified' => '1', 'activated' => '1']);
            $user->load(['appPreference', 'city.country']);
            $user->tokens()->delete();
            $user['token'] = $user->createToken('token')->accessToken;
            DB::commit();
            return Response::success($user);
        } catch (Throwable $e) {
            DB::rollBack();
            throw new HttpResponseException(Response::failure(__('common.error_500_account_verification')));
        }
    }

    public function verify($token, $email)
    {

        try {
            DB::beginTransaction();
            if ((is_null($token) || $token == '')
                || (is_null($email) || $email == '')) {
                return view('failed_account_verified');
            }
            $verifyInfo = PasswordReset::query()
                ->where('token', '=', $token)
                ->where('email', '=', $email)
                ->first();
            $user = User::phoneOrEmail(null, $email)->first();
            if (is_null($verifyInfo) || is_null($user)) {
                return view('failed_account_verified');
            }
            $user->update(['verified' => '1', 'activated' => '1']);
            DB::commit();
            return view('success_account_verified');
        } catch (Throwable $e) {
            DB::rollBack();
            return $e->getMessage();
        }
    }
}
