<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\AssignOrderToDeliveryRequest;
use App\Models\OngoingOrder;
use App\Models\Order;
use App\Models\User;
use App\Services\NotificationService;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class OrderController extends Controller
{

    private $notificationService;


    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        if (config('app.middlewares_enabled') === true) {
            $this->middleware('auth:api');
            $this->middleware('has_permission:view_order')->only(['index', 'export', 'show']);
            $this->middleware('has_permission:update_order')->except('index', 'export', 'show');
        }
    }

    public function index(): JsonResponse
    {
        return Response::success(
            Order::withTrashed()
                ->where('pruned', '=', '0')
                ->numberFilter(request()->query('number_filter'))
                ->userFilter(request()->query('user_id'))
                ->dateFilter(request()->query('date_from'), request()->query('date_to'))
                ->timeFilter(request()->query('time_from'), request()->query('time_to'))
                ->stateFilter(request()->query('state_filter'))
                ->paymentStateFilter(request()->query('payment_state_filter'))
                ->orderByDesc('date')
        );
    }

    public function show($id): JsonResponse
    {
        $order = Order::withTrashed()
            ->findOrFail($id);
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function destroy($id): JsonResponse
    {
        $order = Order::query()->findOrFail($id);
        $order->update([
            'deleted_at' => Carbon::now(),
            'pruned' => true
        ]);
        return Response::success(null);

    }

    public function cancel($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        if ($order->state != 'new' || $order->payment_state != 'awaiting') {
            return Response::failure(__('common.error_cannot_cancel_order'));
        }
        $order->update([
            'state' => 'cancelled'
        ]);
        if ($order->coupon != null) {
            $order->coupon()->update(['used' => false]);
        }
        foreach ($order->orderItems as $orderItem) {
            if ($orderItem->subtracted > 0) {
                $orderItem->product()->increment('quantity', $orderItem->subtracted);
            }
        }
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_cancelled_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function approve($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        $order->update([
            'state' => 'approved'
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_accepted_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function refund($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        $order->update([
            'payment_state' => 'refunded'
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_refunded_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function markAsOngoing($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        if (OngoingOrder::query()
                ->where('delivery_id', '=', $order->delivery_id)
                ->whereHas('order', function ($query) {
                    $query->where('state', '=', 'ongoing');
                })->count() > 0) {
            return Response::failure(__('common.admin_already_has_ongoing'));
        }
        $order->update([
            'state' => 'ongoing'
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_ongoing_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function markAsDelivered($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        OngoingOrder::query()->where('order_id', '=', $order->id)->forceDelete();
        $order->update([
            'state' => 'delivered'
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_delivered_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function ship($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        $order->update([
            'shipping_state' => 'shipped',
            'shipped_date' => Carbon::now(Auth::user()->timezone)
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_shipped_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function paid($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        $order->update([
            'payment_state' => 'paid',
            'deleted_at' => null
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_paid_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function assignToDelivery(AssignOrderToDeliveryRequest $request): JsonResponse
    {
        DB::beginTransaction();
        $delivery = User::query()->findOrFail($request->input('user_id'));
        $order = Order::withTrashed()->findOrFail($request->input('order_id'));
        OngoingOrder::query()->where('order_id', '=', $order->id)->delete();
        OngoingOrder::query()->create([
            'delivery_id' => $delivery->id,
            'order_id' => $order->id
        ]);
        $order->update([
            'state' => 'assigned_to_delivery',
            'delivery_id' => $delivery->id,
            'delivery_assign_date' => Carbon::now($delivery->timezone),
            'warehouse_id' => $request->input('warehouse_id')
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_assigned_to_delivery_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        $this->notificationService->send('common.msg_order_new_order_assigned_title'
            , 'common.msg_order_new_order_assigned_body'
            , $order->delivery
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user.city.country'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }
}
