<?php


namespace App\Services;


use App\Models\Coupon;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Khsing\World\Models\City;
use Khsing\World\Models\Country;

class AuthService
{
    private $geoIpService;
    private $platformSettingService;
    private $accountVerificationService;
    private $couponService;

    public function __construct(GeoIpService $geoIpService
        , PlatformSettingsService            $platformSettingService
        , AccountVerificationService         $accountVerificationService, CouponService $couponService)
    {
        $this->geoIpService = $geoIpService;
        $this->platformSettingService = $platformSettingService;
        $this->accountVerificationService = $accountVerificationService;
        $this->couponService = $couponService;
    }

    public function register(array $fields): ?Model
    {
        $geoIp = $this->geoIpService->get();
        $fields['city_id'] = 32734;
        $fields['timezone'] = 'Asia/Kuwait';

        $city = null;
        $country = null;
        if ($geoIp->city != null) {
            $city = City::query()->where('name', 'LIKE', "%$geoIp->city%")->first();
        } else if ($geoIp->country != null) {
            $country = Country::query()->where('name', 'LIKE', "%$geoIp->country%")->first();
            $city = $country->cities()->inRandomOrder()->first();
        } else {
            $country = Country::query()->where('timezone', 'LIKE', "%$geoIp->timezone%")->first();
            $city = $country->cities()->inRandomOrder()->first();
        }

        if ($city != null && $city->country != null && $city->country->timezone != null) {
            $fields['city_id'] = $city->id;
            $fields['timezone'] = $city->country->timezone;
        } else if ($country != null && $city != null && $country->timezone != null) {
            $fields['city_id'] = $city->id;
            $fields['timezone'] = $country->timezone;
        }

        $fields['role'] = 'Member';
        if ($this->platformSettingService->usersVerification()) {
            $fields['activated'] = $fields['verified'] = false;
            $user = User::query()->create($fields);
            $this->couponService->assignExistingCoupons($user);
            $this->accountVerificationService->create($user);
        } else {
            $fields['activated'] = $fields['verified'] = true;
            $user = User::query()->create($fields);
            $this->couponService->assignExistingCoupons($user);
        }
        return $user;
    }

    public function login($email, $phone, $password): JsonResponse
    {
        $user = User::phoneOrEmail($phone, $email)->first();
        if ($user != null) {
            if ($user->deleted != 0) {
                return Response::failure(__('common.error_invalid_phone'));
            }
            if (Auth::attempt(['email' => $user->email, 'password' => $password])) {
                if ($user->delivery && !$user->accepted) {
                    return Response::failure('common.error_not_accepted');
                }
                if (!$user->verified) {
                    $user->load(['appPreference', 'city.country']);
                    $user->tokens()->delete();
                    $user['token'] = null;
                    return Response::success($user);
                }
                if (!$user->activated) {
                    return Response::failure(__('common.error_not_active'));
                }
            } else {
                $user->increment('access_failed_count');
                return Response::failure(__('common.error_bad_credential'));
            }
        } else {
            return Response::failure(__('common.error_invalid_credential'));
        }
        $user->update([
            'last_login_date' => Carbon::now($user->timezone)->toDateTimeString(),
            'is_logged_in' => true
        ]);
        if ($user->city_id === null) {
            $user->update([
                'timezone' => $this->geoIpService->get()->timezone
            ]);
        }
        if ($user->admin) {
            $user->loadMissing(['roles.permissions']);
        }
        $user->load(['appPreference', 'city.country']);
        $user->tokens()->delete();
        $user['token'] = $user->createToken('token')->accessToken;
        return Response::success($user);
    }

    public function logout(): JsonResponse
    {
        $user = Auth::user();
        $user->tokens()->delete();
        $user->update(['is_logged_in' => false]);
        return Response::success(null);
    }

    public function resetPassword(array $fields): JsonResponse
    {
        $mainUser = Auth::user();
        if (!Hash::check($fields['current_password'], $mainUser->password)) {

            if ($mainUser != null) {
                $mainUser->increment('access_failed_count');
            }
            throw new HttpResponseException(response()->json(['error' => __('common.error_invalid_password')], 400));
        }
        $mainUser->update(['password' => $fields['new_password']]);
        $mainUser->tokens()->delete();
        $mainUser->loadMissing(['city.country']);
        return Response::success($mainUser);
    }

    public function assignExistingCoupons($user)
    {
        $coupons = Coupon::query()
            ->whereDate('expiration', '>=', Carbon::now($user->timezone)->toDateTimeString())
            ->where('active', '=', '1')
            ->get();
        $cs = [];
        foreach ($coupons->groupBy('code') as $key => $value) {
            array_push($cs, [
                'title' => $value[0]['title'],
                'body' => $value[0]['body'],
                'percentage' => $value[0]['percentage'],
                'amount' => $value[0]['amount'],
                'expiration' => $value[0]['expiration'],
                'code_group' => $value[0]['code_group'],
                'code' => $key,
                'for_marketing' => $value[0]['for_marketing'],
                'on_delivery_cost' => $value[0]['on_delivery_cost'],
                'active' => '1',
                'used' => '0',
                'timezone' => $user->timezone,
                'creation' => Carbon::now($user->timezone)->toDateTimeString(),
                'user_id' => $user->id,
            ]);
        }
        Coupon::query()->insert($cs);
    }
}
