<?php


namespace App\Services;


use App\Mail\PasswordResetRequest;
use App\Mail\UserPasswordReset;
use App\Models\PasswordReset;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Str;
use Throwable;

class AccountRecoveryService
{
    private $mailService;
    private $smsService;
    private $hashService;

    public function __construct(MailService $mailService, SmsService $smsService,HashService $hashService)
    {
        $this->mailService = $mailService;
        $this->smsService = $smsService;
        $this->hashService = $hashService;
    }

    public function forgotPassword($email, $phone): JsonResponse
    {
        $user = User::phoneOrEmail($phone, $email)->first();
        if ($user == null)
            return Response::failure(__('common.error_invalid_credential'));
        if (!$user->activated)
            return Response::failure(__('common.error_not_active'));
        $latestAttempt = PasswordReset::query()->where('email', '=', $user->email)->latest()->first();
        if ($latestAttempt != null && Carbon::now()->diffInRealMinutes(Carbon::parse($latestAttempt->created_at), true) < 2) {
            throw new HttpResponseException(response()->json(['error' => __('common.error_wait_2_min')], 400));
        }
        PasswordReset::query()->where('email', '=', $user->email)->forceDelete();
        $token = Str::random(50) . uniqid("", true) . Str::random(50) . time();
        $token = preg_replace('/[^a-zA-Z0-9_ -]/s', '', $token);
        $verificationCode = $this->hashService->numericHash(time(), 6);
        PasswordReset::query()->create([
            'email' => $user->email,
            'token' => $token,
            'verification_code' => $verificationCode,
            'created_at' => Carbon::now($user->timezone)
        ]);
        $url = config('app.url') . '/api/resetForgotPassword?';
        $url = $url . 'token=' . $token;
        $url = $url . '&';
        $url = $url . 'email=' . $user->email;
        $this->mailService->sendToUser($user, new PasswordResetRequest($url, $user->fullname,$verificationCode));
        $this->smsService->send($user->phone, $verificationCode, $user->locale);
        return Response::success(['message' => __('common.msg_verification_sent')]);
    }

    public function resetForgotPassword($token, $email)
    {
        try {
            DB::beginTransaction();
            if ((is_null($token) || $token == '')
                || (is_null($email) || $email == '')) {
                return view('failed_password_reset');
            }
            $verifyInfo = PasswordReset::query()
                ->where('token', '=', $token)
                ->where('email', '=', $email)
                ->first();
            $user = User::phoneOrEmail(null, $email)->first();
            if (is_null($verifyInfo) || is_null($user)) {
                return view('failed_password_reset');
            }
            $password = Str::random(10);
            $user->update(['password' => $password]);
            $verifyInfo->forceDelete();
            $this->mailService->sendToUser($user, new UserPasswordReset($user, $password));
            DB::commit();
            return view('success_password_reset');
        } catch (Throwable $e) {
            DB::rollBack();
            return $e->getMessage();
        }
    }

    public function resetForgotPasswordByCode($user, $code, $newPassword): JsonResponse
    {
        DB::beginTransaction();
        $verifyInfo = PasswordReset::query()
            ->where('verification_code', '=', $code)
            ->where('email', '=', $user->email)
            ->first();
        if (is_null($verifyInfo)) {
            throw new HttpResponseException(Response::failure(__('common.error_500_account_verification')));
        }

        $user->update([
            'verified' => '1',
            'activated' => '1',
            'password' => $newPassword
        ]);
        PasswordReset::query()->where('id', '=', $verifyInfo->id)->forceDelete();
        $user->load(['appPreference', 'city.country']);
        $user->tokens()->delete();
        $user['token'] = $user->createToken('token')->accessToken;
        DB::commit();
        return Response::success($user);
    }
}
