<?php
/** @noinspection DuplicatedCode */
/** @noinspection PhpUnused */
/** @noinspection PhpUndefinedMethodInspection */
/** @noinspection PhpUndefinedFieldInspection */

namespace App\Http\Controllers\Delivery;

use App\Http\Controllers\Controller;
use App\Http\Requests\OrderDeliveredRequest;
use App\Models\OngoingOrder;
use App\Models\Order;
use App\Services\NotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class DeliveryController extends Controller
{
    private $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        $this->middleware('auth:api');

        $this->middleware('is_delivery');

        $this->middleware('is_active');
    }

    public function index(Request $request): JsonResponse
    {
        $mainUser = Auth::user();
        if ($request->query('type') === 'delivered') {
            return Response::success(Order::query()
                ->where('delivery_id', '=', $mainUser->id)
                ->where('state', '=', 'delivered')
                ->with([
                    'user.city.country'
                    , 'delivery'
                    , 'coupon'
                    , 'billingAddress'
                    , 'shippingAddress'
                    , 'seller'
                    , 'payment'
                    , 'deliveryOption'
                    , 'orderItems.product'
                    , 'warehouse'
                    , 'address'
                ])
                ->latest());
        } else if ($request->query('type') === 'pending') {
            return Response::success(OngoingOrder::query()
                ->where('delivery_id', '=', $mainUser->id)
                ->whereHas('order', function ($query) {
                    $query->where('state', '=', 'assigned_to_delivery');
                })
                ->with([
                    'order.user'
                    , 'delivery'
                    , 'order.coupon'
                    , 'order.billingAddress'
                    , 'order.shippingAddress'
                    , 'order.seller'
                    , 'order.payment'
                    , 'order.deliveryOption'
                    , 'order.orderItems.product'
                    , 'order.warehouse'
                    , 'order.address'
                ])
                ->latest());
        } else if ($request->query('type') === 'pending_or_ongoing') {
            return Response::success(OngoingOrder::query()
                ->where('delivery_id', '=', $mainUser->id)
                ->whereHas('order', function ($query) {
                    $query->where(function ($query) {
                        $query->where('state', '=', 'assigned_to_delivery')->orWhere('state', '=', 'ongoing');
                    });
                })
                ->with([
                    'order.user'
                    , 'delivery'
                    , 'order.coupon'
                    , 'order.billingAddress'
                    , 'order.shippingAddress'
                    , 'order.seller'
                    , 'order.payment'
                    , 'order.deliveryOption'
                    , 'order.orderItems.product'
                    , 'order.warehouse'
                    , 'order.address'
                ])
                ->latest());
        } else if ($request->query('type') === 'ongoing') {
            return Response::success(OngoingOrder::query()
                ->where('delivery_id', '=', $mainUser->id)
                ->whereHas('order', function ($query) {
                    $query->where('state', '=', 'ongoing');
                })
                ->with([
                    'order.user'
                    , 'delivery'
                    , 'order.coupon'
                    , 'order.billingAddress'
                    , 'order.shippingAddress'
                    , 'order.seller'
                    , 'order.payment'
                    , 'order.deliveryOption'
                    , 'order.orderItems.product'
                    , 'order.warehouse'
                    , 'order.address'
                ])
                ->latest());
        } else {
            return Response::failure(__('common.error_invalid_request'));
        }
    }

    public function orderDelivered(OrderDeliveredRequest $request): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($request->input('order_id'));
        if ($request->input('delivery_code') !== $order->delivery_code) {
            return Response::failure(__('common.error_invalid_delivery_code'));
        }
        $input['state'] = 'delivered';
        if ($order->payment->type == 'offline') {
            $input['non_payment_reason'] = $request->input('non_payment_reason');
            if ($request->input('paid') == '0') {
                $input['payment_state'] = 'cancelled';
            }
            if ($request->input('paid') == '1') {
                $input['payment_state'] = 'paid';
            }
        }
        OngoingOrder::query()->where('order_id', '=', $order->id)->forceDelete();
        $order->update($input);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_delivered_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }

    public function markOnGoing($id): JsonResponse
    {
        DB::beginTransaction();
        $order = Order::withTrashed()
            ->findOrFail($id);
        if (OngoingOrder::query()
                ->where('delivery_id', '=', Auth::user()->id)
                ->whereHas('order', function ($query) {
                    $query->where('state', '=', 'ongoing');
                })->count() > 0) {
            return Response::failure(__('common.delivery_already_has_ongoing'));
        }
        $order->update([
            'state' => 'ongoing'
        ]);
        DB::commit();
        $this->notificationService->send('common.msg_order_status_title'
            , 'common.msg_order_ongoing_body'
            , $order->user
            , ['order' => $order->number]
            , NotificationService::ORDERS
        );
        return Response::success($order->load([
            'user'
            , 'delivery'
            , 'coupon'
            , 'billingAddress'
            , 'shippingAddress'
            , 'seller'
            , 'payment'
            , 'deliveryOption'
            , 'orderItems.product'
            , 'warehouse'
            , 'address'
        ]));
    }
}
